import wp = require('whatsapp-web.js');
const Client = wp.Client;
const LocalAuth = wp.LocalAuth;
import Message = wp.Message;

import * as qrcode from 'qrcode-terminal';
import QRCode from 'qrcode';
import { createConversationContext, limitConversationHistory, clearChatSession, getSessionStats, ConversationMessage } from './openai-simple';
import { sendMessageToGemini, reloadConfiguration } from './gemini-adapter';
import dotenv from 'dotenv';
import express from 'express';
import { createServer } from 'http';
import { Server } from 'socket.io';
import path from 'path';
import fs from 'fs';

// Carrega variáveis de ambiente
dotenv.config();

// Configuração do Servidor Web
const app = express();
const httpServer = createServer(app);
const io = new Server(httpServer);
const PORT = process.env.PORT || 3000;

// Serve arquivos estáticos (Ajuste para funcionar tanto em Dev quanto em Prod/Dist)
const publicPath = path.join(__dirname, 'public');
// Se não achar no dist, tenta na raiz (para dev)
if (!fs.existsSync(publicPath)) {
    app.use(express.static(path.join(__dirname, '../public')));
} else {
    app.use(express.static(publicPath));
}
app.use(express.json());

// Endpoint de Configuração Inicial (Licença + API Key)
app.post('/configure', (req, res) => {
    const { licenseKey, apiKey, companyName } = req.body;

    // Salva no .env (para o sistema)
    let envContent = `PORT=3000\n`;
    if (apiKey) envContent += `GEMINI_API_KEY=${apiKey}\n`;
    fs.writeFileSync(path.join(__dirname, '../.env'), envContent);

    // Salva no JSON de configuração
    const configPath = path.join(__dirname, '../config/client_settings.json');
    const configData = {
        license_key: licenseKey || "DEMO-MODE",
        gemini_api_key: apiKey || "",
        company_name: companyName || "Minha Empresa"
    };
    fs.writeFileSync(configPath, JSON.stringify(configData, null, 2));

    // Recarrega configuração na memória
    reloadConfiguration();

    res.json({ success: true, message: "OrionBot configurado com sucesso!" });
});

const client = new Client({
    authStrategy: new LocalAuth({ dataPath: './session' }),
    puppeteer: {
        headless: true,
        args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu']
    },
    webVersionCache: {
        type: 'remote',
        remotePath: 'https://raw.githubusercontent.com/wppconnect-team/wa-version/main/html/2.2412.54.html'
    }
});

httpServer.listen(PORT, () => {
    console.log(`🌐 OrionBot Client rodando na porta ${PORT}`);
    console.log(`🔗 Interface de Gestão: http://localhost:${PORT}`);
});

io.on('connection', (socket) => {
    socket.emit('status', 'Aguardando Leitura do QR Code...');
});

const conversationHistory = new Map<string, ConversationMessage[]>();

async function processMessage(message: Message): Promise<void> {
    try {
        const chatId = message.from;
        const messageText = message.body;
        const contactName = (message as any)._data?.notifyName || "Cliente";

        if (message.fromMe) return;
        if (message.from.includes('@g.us')) return; // Ignora grupos por padrão

        let history = conversationHistory.get(chatId) || [];
        history = limitConversationHistory(history, 20);
        history.push({ role: "user", content: messageText });

        const context = createConversationContext(
            chatId,
            contactName,
            message.type,
            message.from.includes('@g.us')
        );

        let response;

        // Suporte a Áudio (Multimodal)
        if (message.hasMedia && (message.type === 'ptt' || message.type === 'audio')) {
            console.log(`🎤 Áudio recebido de ${contactName}`);
            try {
                const media = await message.downloadMedia();
                if (media) {
                    response = await sendMessageToGemini(
                        messageText || "",
                        context,
                        history,
                        { mimeType: media.mimetype, data: media.data }
                    );
                } else {
                    response = { success: false, text: "Erro na recepção do áudio." };
                }
            } catch (e) {
                console.error("Erro audio:", e);
                response = { success: false, text: "Não entendi o áudio." };
            }
        } else {
            response = await sendMessageToGemini(messageText, context, history);
        }

        if (response.success) {
            history.push({ role: "assistant", content: response.text });
            conversationHistory.set(chatId, history);
            await message.reply(response.text);
        }

    } catch (error) {
        console.error('Erro processMessage:', error);
    }
}

async function handleSpecialCommands(message: Message): Promise<boolean> {
    const text = message.body.toLowerCase().trim();
    if (text === '!reset' || text === '!limpar') {
        conversationHistory.delete(message.from);
        await message.reply('✅ Memória da conversa limpa.');
        return true;
    }
    if (text === '!update' || text === '!atualizar') {
        reloadConfiguration();
        await message.reply('🔄 Configurações e Cérebro recarregados da nuvem.');
        return true;
    }
    return false;
}

client.once('ready', () => {
    console.log('✅ OrionBot Client Conectado ao WhatsApp!');
    io.emit('ready', true);
    io.emit('status', 'Online e Operando');
});

client.on('qr', (qr) => {
    qrcode.generate(qr, { small: true });
    QRCode.toDataURL(qr, (err, url) => {
        if (!err) {
            io.emit('qr', url);
            io.emit('status', 'Escaneie o QR Code');
        }
    });
});

client.on('message_create', async (message) => {
    if (!await handleSpecialCommands(message)) {
        await processMessage(message);
    }
});

client.initialize();
